/*
  Name: T7info
  Copyright: Freeware, use as you please
  Author: Tomi Liljemark (firstname.surname@gmail.com)
  Created: 2007-06-08
  Modified: 2007-10-03
  Compiler: Dev-C++ v4.9.9.2 (shouldn't matter)
  Description: Reads and prints out information from a Saab Trionic 7 ECU binary.
*/

#include <stdio.h>
#include <stdlib.h>

#define  RELEASE_VERSION "0.60"
#define  RELEASE_DATE    "2007-10-03"

struct head_t
{
      unsigned char length;
      unsigned char id;
      union
      {
           unsigned char string[130];
           unsigned int hex32;
           unsigned short hex16;
           unsigned char hex8;
      } data;
      unsigned int addr;
};

struct csum_area_t
{
       long addr;
       short length;
};


unsigned char binary[512*1024];      /* Store to whole 512 kB binary in RAM */

/* Private variables */
struct head_t header_info[30];
char header_info_count;
struct csum_area_t csum_area[16];
unsigned int csum;
int binary_length;
unsigned char motorola_detected;

struct
{
      unsigned int fb;
      unsigned int f2;
      unsigned int misc;
      unsigned int area_70000;
} checksum_addr;

struct
{
      unsigned int fb;
      unsigned int f2;
      unsigned int misc;
      unsigned int area_70000;
} checksum_calc;

struct
{
      unsigned int fb;
      unsigned int f2;
      unsigned int misc;
      unsigned int area_70000;
} checksum_read;

/* Private functions */
unsigned int search_csum_area_start(unsigned char *bin);
int search_70000_area_length(unsigned char *bin);
void search_csum_areas(unsigned char *bin, unsigned int area_start);
unsigned int calc_csum(unsigned char *bin, unsigned int start, int length);
unsigned int calc_csum2(unsigned char *bin, unsigned int start, int length);
unsigned int calc_csum_f2(unsigned char *bin, unsigned int start, int length);
void read_header(unsigned char *bin, unsigned int end_addr);
unsigned int find_header_field_32(unsigned char type);
char *find_header_field_string(unsigned char type);
unsigned int find_70000_checksum( unsigned char* bin, int addr );

int load_file(const char *filename, unsigned char *data);
void write_new_header_codes(const char *filename, const char *vin, const char *immo);
int check_vin( const char* vin );
int find_header_field_addr(unsigned char *bin, unsigned char type);
void write_new_checksums(const char *filename, unsigned int fb,
                                               unsigned int f2,
                                               unsigned int misc,
                                               unsigned int area_70000);

int main(int argc, char *argv[])
{
    struct head_t header_field;
    long addr;
    unsigned char i, byte, data[255];
    unsigned char *bin;
    char all_ascii;
    unsigned short checksum16;
    unsigned int checksum32, checksum, start, stop;
    char new_vin[30], new_immo[30];

    header_info_count = 0;
    bin = binary;
    motorola_detected = 0;

    printf("T7info v%s - Read information from a Saab Trionic 7 ECU\r\n"
           "by Tomi Liljemark %s\r\n\r\n", RELEASE_VERSION, RELEASE_DATE);


    if( argc < 2 || 
        !strcmp(argv[1], "?") || !strcmp(argv[1], "-?") || !strcmp(argv[1], "/?") || 
        !strcmp(argv[1], "-h") || !strcmp(argv[1], "/h") || !strcmp(argv[1], "--help") ||
        !strcmp(argv[1], "/help") )
    {
        printf("Usage: %s [--csum] [--marry] <filename.bin>\r\n", argv[0]);
        return -1;
    }

    if( load_file( argv[argc-1], bin ) == -1 )
    {
        printf("Error loading file %s!\r\n", argv[argc-1]);
        return -1;
    }


    /* Trionic 7 binaries are 512 kB, Trionic 5 are 256 kB */    
    byte = 0xFF;
    addr = binary_length-1;

    if( argc == 3 &&
        (!strcmp(argv[1], "-m") || !strcmp(argv[1], "/m") || !strcmp(argv[1], "--marry") ||
        !strcmp(argv[1], "/marry")) )
    {
        if( binary_length == 0x70100 )
        {
            printf("Error: you can't marry a \"TIS\" binary.\n");
            return -1;
        }
        else if( motorola_detected )
        {
            printf("Error: you can't marry a Motorola byte-order file (not implemented yet).\n");
            return -1;
        }
        
        read_header( bin, addr );

        printf("Current values\r\n"
               "--------------\r\n");        
        printf("Vehicle Identification Number: %s\r\n", find_header_field_string( 0x90 ) );
        printf("Immobilizer code             : %s\r\n", find_header_field_string( 0x92 ) );

        printf("\r\nEnter new values\r\n"
               "----------------\r\n");
               
        printf("Vehicle Identification Number: ");
        scanf("%s", new_vin);
        printf("Immobilizer code             : ");
        scanf("%s", new_immo);

        printf("\r\n");        
        if( strlen(new_vin) != 17 ) printf("Warning: VIN  should be 17 characters long!\r\n");
        if( strncmp(new_vin, "YS3", 3) && strncmp(new_vin, "JF4", 3) && strncmp(new_vin, "5S3", 3) ) 
            printf("Warning: VIN does not match with Saab!\r\n");
        if( strlen(new_immo) != 15 ) printf("Warning: Immobilizer code should be 15 characters long!\r\n");
        if( check_vin( new_vin ) ) printf("Warning: VIN check symbol incorrect!\r\n");
        
        printf("\r\nAre you sure you want to change these codes [y/N] ? ");
        scanf("%s", data);
        if( data[0] == 'y' || data[0] == 'Y' )
        {
            write_new_header_codes( argv[argc-1], new_vin, new_immo );
            printf("Changed.\r\n");
        }
        else printf("Aborted, nothing written.\r\n");
        
        
        //write_new_header_codes( argv[argc-1], "VIN", "IMMO" );
        return 0;
    }

    if( argc == 3 &&
        (!strcmp(argv[1], "-c") || !strcmp(argv[1], "/c") || !strcmp(argv[1], "--csum") ||
        !strcmp(argv[1], "/csum")) )
    {
        read_header( bin, addr );
        addr = search_csum_area_start(bin);
        if( addr != -1 )
        {
            //printf("Checksum area start address 0x%05lX.\r\n", addr);
            search_csum_areas( bin, addr );
        }

        checksum32 = 0;
        for( i = 0; i < 16; i++ )
        {
             checksum = calc_csum( bin, csum_area[i].addr, csum_area[i].length );
             //printf("Addr 0x%05lX\tLength 0x%03lX\tChecksum 0x%08X\r\n", csum_area[i].addr, csum_area[i].length, checksum);
             checksum32 += checksum;
        }
        checksum_read.misc = csum;
        checksum_calc.misc = checksum32;
        printf("\r\n"
               "Calculated checksum Misc : 0x%08X - %s\r\n", 
               checksum32, checksum32 == csum ? "PASS" : "FAIL");


        if( binary_length == 0x80000 )
        {
            start = 0x70000;
            stop  = search_70000_area_length( bin );
            if( stop != -1 )
            {
                checksum = calc_csum( bin, start, stop );
                checksum_calc.area_70000 = checksum;
                checksum_read.area_70000 = find_70000_checksum( bin, stop+0x70000 );
                printf("Calculated checksum 70000: 0x%08X - %s\r\n", checksum,
                              checksum == checksum_read.area_70000 ? "PASS" : "FAIL");
            }
            else
            {
                checksum_read.area_70000 = 0xFFFFFFFF;
            }
        }
        else
        {
            checksum_read.area_70000 = 0xFFFFFFFF;
        }

        start = find_header_field_32( 0xFD );
        stop = find_header_field_32( 0xFE );
        checksum_read.f2 = find_header_field_32( 0xF2 );
        if( checksum_read.f2 != 0xFFFFFFFF )
        {
            checksum = calc_csum_f2( bin, start, stop-start );
            checksum_calc.f2 = checksum;
            printf("Calculated checksum 0xF2 : 0x%08X - %s\r\n", checksum,
                          checksum == checksum_read.f2 ? "PASS" : "FAIL");
        }
        
        start = find_header_field_32( 0xFD );
        stop = find_header_field_32( 0xFE );
        checksum = calc_csum( bin, start, stop-start );
        checksum_calc.fb = checksum;
        checksum_read.fb = find_header_field_32( 0xFB );
        printf("Calculated checksum 0xFB : 0x%08X - %s\r\n", checksum,
                      checksum == checksum_read.fb ? "PASS" : "FAIL");


        checksum_addr.fb = find_header_field_addr( bin, 0xFB );
        checksum_addr.f2 = find_header_field_addr( bin, 0xF2 );
        
        //printf("0x%05X, 0x%05X, 0x%05X, 0x%05X\r\n", 
        //                checksum_addr.misc, checksum_addr.fb, checksum_addr.f2, checksum_addr.area_70000);
        
        if( (checksum_calc.f2 != checksum_read.f2 &&
             checksum_read.f2 != 0xFFFFFFFF ) ||
            checksum_calc.fb != checksum_read.fb ||
            checksum_calc.misc != checksum_read.misc ||
            (checksum_calc.area_70000 != checksum_read.area_70000 &&
             checksum_read.area_70000 != 0xFFFFFFFF) )
        {
            printf( "\r\nWarning: One or more checksums did not match!\r\n\r\n"
                    "Do you want to recalculate checksums [y/N] ? ");
            scanf("%s", data );
            if( data[0] == 'y' || data[0] == 'Y' )
            {
                if( motorola_detected )
                {
                    printf("Error: you can't correct checksums on a Motorola byte-order file (not implemented yet).\n");
                    return -1;
                }
                write_new_checksums( argv[argc-1], checksum_calc.fb,
                                                   checksum_calc.f2,
                                                   checksum_calc.misc,
                                                   checksum_calc.area_70000 );
                printf("Changed.\r\n");
            }
            else printf("Aborted, nothing written.\r\n");
            
            
        }
        
        return 0;
    }
    


    /* Reads backwards from the end of the binary */
    printf("Length  ID    Data\r\n");
    while( 1 )
    {
        /* The first byte is the length of the data */
        header_info[header_info_count].length = *(bin+addr);
        if( header_info[header_info_count].length == 0x00 ||
            header_info[header_info_count].length == 0xFF ) break;
        addr--;
        /* Second byte is an ID field */
        header_info[header_info_count].id = *(bin+addr);
        addr--;
        printf("%6d  0x%02x  [", header_info[header_info_count].length, 
                                 header_info[header_info_count].id);
        /* The actual data */
        all_ascii = 1;
        for( i = 0; i < header_info[header_info_count].length; i++ )
        {
             data[i] = *(bin+addr);
             addr--;
             if( !(data[i] >= 32 && data[i] <= 126) ) all_ascii = 0;
        }
        data[header_info[header_info_count].length] = 0;
        memcpy( header_info[header_info_count].data.string, 
                data, 
                header_info[header_info_count].length+1 );

        /* If all characters are printable, print in ASCII otherwise use Hex */
        if( all_ascii == 1 ) printf("%s", header_info[header_info_count].data.string);
        else
        {
            for( i = 0; i < header_info[header_info_count].length; i++ )
                 printf("%02x", header_info[header_info_count].data.string[i]);
        }
        printf("]\r\n");
        header_info_count++;
    }

    return 0;
}

int load_file(const char *filename, unsigned char *data)
{
    FILE *bin;
    int i;
    size_t read_bytes;
    unsigned char temp;

    read_bytes = 0;
    bin = fopen( filename, "rb" );

    if( bin != NULL )
    {    
        fseek( bin, 0, SEEK_SET );
        read_bytes = fread( data, sizeof(unsigned char), 512*1024, bin );
        fclose( bin );
    }
    else
    {
        printf("Error opening: %s\r\n", filename);
    }

    if( read_bytes == 256*1024 )
    {
        printf("Error: is this a Trionic 5 ECU binary?\n");
    }
    else if( read_bytes == 512*1024 || read_bytes == 0x70100 )
    {
        // Convert Motorola byte-order to Intel byte-order (just in RAM)
        if( data[0] == 0xFF && data[1] == 0xFF && data[2] == 0xFC && data[3] == 0xEF )
        {
            motorola_detected = 1;
            for( i = 0; i < read_bytes; i += 2 )
            {
                temp = data[i];
                data[i] = data[i+1];
                data[i+1] = temp;
            }
            printf("Note: Motorola byte-order detected.\n");
        }
        
    }
    
    binary_length = read_bytes;
    
    if( read_bytes == 0x80000 ) return 0;
    else if( read_bytes == 0x70100 ) return 1;
    else return -1;
}


unsigned int search_csum_area_start(unsigned char *bin)
{
       const char sequence[24] = { 0x48,0xE7,0x00,0x3C,0x24,0x7C,0x00,0xF0,
                                   0x00,0x00,0x26,0x7C,0x00,0x00,0x00,0x00,
                                   0x28,0x7C,0x00,0xF0,0x00,0x00,0x2A,0x7C };
       const char seq_mask[24] = {    1,   1,   1,   1,   1,   1,   1,   1,
                                      0,   0,   1,   1,   1,   0,   0,   0,   
                                      1,   1,   1,   1,   0,   0,   1,   1 };

     char data;
     int i, k, max;
     int addr;
     
     addr = 0;
     i = 0;
     max = 0;
     while( addr < (binary_length-1) )
     {
            data = *(bin+addr);
            if( data == sequence[i] || seq_mask[i] == 0 )
            {
                i++;
            }
            else
            {
                if( i > max ) max = i;
                i = 0;
            }
            if( i == 24 ) break;
            addr++;            
     }
     if( i == 24 )
     {
         return (addr - 23);
     }
     else
     {
         printf("Error: Can't find checksum area. Max match %d\r\n", max);
         return 0x80000;
     }
}

int search_70000_area_length(unsigned char *bin)
{
     const unsigned char sequence[24] = {0x20, 0x3c, 0x00, 0x00, 0x11, 0x52, 0x2F, 
                                         0x00, 0x20, 0x3C, 0x00, 0x00, 0x09, 0xD0, 
                                         0x2F, 0x00, 0x20, 0x3C, 0x00, 0x00, 0x00, 
                                         0xCC, 0xD0, 0x9F };
     const char seq_mask[24] = {            1,    1,    0,    0,    0,    0,    1,
                                            1,    1,    1,    0,    0,    0,    0,    
                                            1,    1,    1,    1,    0,    0,    0,    
                                            0,    1,    1 };

     const char places[3] = { 2, 10, 18 };
     int i, k, max, length, addr, temp;
     
     addr = 0;
     i = 0;
     max = 0;
     while( addr < (binary_length-1) )
     {
            if( *(bin+addr) == sequence[i] || seq_mask[i] == 0 )
            {
                i++;
            }
            else
            {
                if( i > max ) max = i;
                i = 0;
            }
            if( i == 24 ) break;
            addr++;
     }
     if( i == 24 )
     {
        length = 0;
        addr -= 23;
        //printf("70000 area: 0x%05X\r\n", addr);

        for( i = 0; i < 3; i++ )
        {
            //printf("0x%05x - ", addr+places[i] );
            temp = (unsigned int)( *(bin+addr+places[i]+0) << 24 | 
                                   *(bin+addr+places[i]+1) << 16 | 
                                   *(bin+addr+places[i]+2) << 8 | 
                                   *(bin+addr+places[i]+3));
            //printf("0x%05x\r\n", temp);
            length += temp;
        }
         
         return length;
     }
     else
     {
         //printf("Error: Can't find 70000 area length. Max match %d\r\n", max);
         return -1;
     }
}

unsigned int find_70000_checksum( unsigned char* bin, int addr )
{
    unsigned int temp;

    temp = (unsigned int)( *(bin+addr+0) << 24 | 
                           *(bin+addr+1) << 16 | 
                           *(bin+addr+2) << 8 | 
                           *(bin+addr+3));
    //printf("7000 checksum: 0x%08X (given addr 0x%05X)\r\n", temp, addr);
    checksum_addr.area_70000 = addr;
    return temp;
}

void search_csum_areas(unsigned char *bin, unsigned int area_start)
{
     unsigned char data[4], area_number;
     unsigned int addr, base_addr, ltemp, csum_addr;
     short csum_length;
     
     if( area_start > (binary_length-1) ) return;
     addr = area_start + 22;
     area_number = 0;
     
     while( addr < (binary_length-1) )
     {
            data[0] = *(bin+addr+0);
            data[1] = *(bin+addr+1);
            addr += 2;
            if( data[0] == 0x48 )
            {
                switch( data[1] )
                {
                    case 0x6D:
                        data[0] = *(bin+addr+0);
                        data[1] = *(bin+addr+1);
                        addr += 2;
                        csum_addr = base_addr + (unsigned int)(data[0] << 8 | data[1]);
                        //printf("Addr 0x%05X\r\n",  csum_addr );
                        csum_area[area_number].addr = csum_addr;
                        area_number++;
                        break;
                    case 0x78:
                        data[0] = *(bin+addr+0);
                        data[1] = *(bin+addr+1);
                        addr += 2;
                        csum_length = (short)(data[0] << 8 | data[1]);
                        //printf("Length 0x%03lX\t",  csum_length );
                        csum_area[area_number].length = csum_length;
                        break;
                    case 0x79:
                        data[0] = *(bin+addr+0);
                        data[1] = *(bin+addr+1);
                        data[2] = *(bin+addr+2);
                        data[3] = *(bin+addr+3);
                        addr += 4;
                        csum_addr = (unsigned int)(data[0] << 24 | data[1] << 16 | data[2] << 8 | data[3]);
                        //printf("Addr 0x%05lX\r\n",  csum_addr );
                        csum_area[area_number].addr = csum_addr;
                        area_number++;
                        break;
                    default:
                        break;
                }
            }
            else if( data[0] == 0x2A && data[1] == 0x7C )
            {
                data[0] = *(bin+addr+0);
                data[1] = *(bin+addr+1);
                data[2] = *(bin+addr+2);
                data[3] = *(bin+addr+3);
                addr += 4;
                ltemp = (unsigned int)(data[0] << 24 | data[1] << 16 | data[2] << 8 | data[3]);
                if( ltemp < 0xF00000L )
                {
                    base_addr = ltemp;
                    //printf("Base addr 0x%05lX\r\n",  base_addr);
                }
            }
            else if( data[0] == 0xB0 && data[1] == 0xB9 )
            {
                data[0] = *(bin+addr+0);
                data[1] = *(bin+addr+1);
                data[2] = *(bin+addr+2);
                data[3] = *(bin+addr+3);
                addr += 4;
                csum_addr = (unsigned int)(data[0] << 24 | data[1] << 16 | data[2] << 8 | data[3]);
                //printf("Checksum Addr 0x%05lX\r\n",  csum_addr );
                checksum_addr.misc = csum_addr;
                //printf("Checksum area number: %d\r\n", area_number);

                data[0] = *(bin+csum_addr+0);
                data[1] = *(bin+csum_addr+1);
                data[2] = *(bin+csum_addr+2);
                data[3] = *(bin+csum_addr+3);
                csum = (unsigned int)(data[0] << 24 | data[1] << 16 | data[2] << 8 | data[3]);
                printf("Checksum Misc : 0x%08X\r\n", csum);
                
                ltemp = search_70000_area_length( bin );
                if( ltemp != -1 )
                {
                    printf("Checksum 70000: 0x%08X\r\n", find_70000_checksum( bin, ltemp + 0x70000 ) );
                }
                ltemp = find_header_field_32( 0xF2 );
                if( ltemp != 0xFFFFFFFF ) printf("Checksum 0xF2 : 0x%08X\r\n", ltemp);
                printf("Checksum 0xFB : 0x%08X\r\n", find_header_field_32( 0xFB ));
                break;
            }
            
     }
}


unsigned int calc_csum(unsigned char *bin, unsigned int start, int length)
{
    unsigned int addr;
    unsigned char data[4];
    unsigned char checksum8;
    unsigned int checksum;
    int count;
     
    addr = start;
    checksum = 0;
    count = 0;

    while( count < (length >> 2) && addr < 0x7FFFF )
    {
        data[0] = *(bin+addr+0);
        data[1] = *(bin+addr+1);
        data[2] = *(bin+addr+2);
        data[3] = *(bin+addr+3);
        checksum += (unsigned int)(data[0] << 24 | data[1] << 16 | data[2] << 8 | data[3]);
        addr += 4;
        count++;
    }     
    count = count << 2;
    checksum8 = 0;

    while( count < length && addr < (binary_length-1) )
    {
        data[0] = *(bin+addr);
        checksum8 += data[0];
        addr++;
        count++;
    }

    checksum += checksum8;
    
    return checksum;
}

unsigned int calc_csum_f2(unsigned char *bin, unsigned int start, int length)
{
    unsigned int addr;
    const int xor_table[8] = { 0x81184224, 0x24421881, 0xc33c6666, 0x3cc3c3c3,
                               0x11882244, 0x18241824, 0x84211248, 0x12345678 };
    unsigned char data[4];
    unsigned char xor_count;
    unsigned int temp, checksum;
    int count;
     
    addr = start;
    checksum = 0;
    count = 0;
    xor_count = 1;

    while( count < length && addr < (binary_length-1) )
    {
        data[0] = *(bin+addr+0);
        data[1] = *(bin+addr+1);
        data[2] = *(bin+addr+2);
        data[3] = *(bin+addr+3);
        temp = (unsigned int)(data[0] << 24 | data[1] << 16 | data[2] << 8 | data[3]);
        checksum += temp ^ xor_table[xor_count++];
        if( xor_count > 7 ) xor_count = 0;
        addr += 4;
        count += 4;
    }     

    checksum ^= 0x40314081;
    checksum -= 0x7FEFDFD0;

    return checksum;
}

unsigned int calc_csum2(unsigned char *bin, unsigned int start, int length)
{
    unsigned int addr;
    unsigned char data[4];
    unsigned char checksum8;
    unsigned int checksum;
    int count;
     
    addr = start;
    checksum = 0;
    count = 0;

    while( count < length && addr < (binary_length-1) )
    {
        data[0] = *(bin+addr+0);
        data[1] = *(bin+addr+1);
        data[2] = *(bin+addr+2);
        data[3] = *(bin+addr+3);
        checksum += (unsigned int)(data[0] << 24 | data[1] << 16 | data[2] << 8 | data[3]);
        addr += 4;
        count+= 4;
    }     

    return checksum;
}

void read_header(unsigned char *bin, unsigned int end_addr)
{
     unsigned char byte;
     unsigned char data[255];
     unsigned int addr;
     int i;
     
     addr = end_addr;
    /* Reads backwards from the end of the binary */
    while( 1 )
    {
        /* The first byte is the length of the data */
        header_info[header_info_count].length = *(bin+addr);
        if( header_info[header_info_count].length == 0x00 ||
            header_info[header_info_count].length == 0xFF ) break;
        addr--;
        /* Second byte is an ID field */
        header_info[header_info_count].id = *(bin+addr);
        addr--;

        header_info[header_info_count].addr = addr;
        /* The actual data */
        for( i = 0; i < header_info[header_info_count].length; i++ )
        {
             data[i] = *(bin+addr);
             addr--;
        }
        data[header_info[header_info_count].length] = 0;
        memcpy( header_info[header_info_count].data.string, 
                data, 
                header_info[header_info_count].length+1 );

        header_info_count++;
    }
}

unsigned int find_header_field_32(unsigned char type)
{
    unsigned char i;
     
    for( i = 0; i < header_info_count; i++ )
    {
         if( header_info[i].id == type )
         {
             return (unsigned int)( header_info[i].data.string[0] << 24 | 
                                    header_info[i].data.string[1] << 16 | 
                                    header_info[i].data.string[2] << 8 | 
                                    header_info[i].data.string[3]);
         }
    }
    if( type != 0xF2) printf("Error: Did not find field from header! (0x%02X)\r\n", type);
    return 0xFFFFFFFF;
}

char *find_header_field_string(unsigned char type)
{
    unsigned char i, last_found_index;
     
    last_found_index = 255;
    for( i = 0; i < header_info_count; i++ )
    {
         if( header_info[i].id == type )
         {
             last_found_index = i;
         }
    }
    if( last_found_index != 255 )
        return (char*)header_info[last_found_index].data.string;
    else    
        return NULL;
}

int find_header_field_addr(unsigned char *bin, unsigned char type)
{
    unsigned char byte, id;
    unsigned int addr;
    short length;
    int i;
     
     addr = binary_length-1;
    /* Reads backwards from the end of the binary */
    while( addr > (binary_length-0x300) )
    {
        /* The first byte is the length of the data */
        length = *(bin+addr);
        if( length == 0x00 || length == 0xFF ) break;
        addr--;
        /* Second byte is an ID field */
        id = *(bin+addr);
        addr--;

        if( type == id ) return addr;
        addr -= length;
    }
    
    return 0x80000;
}

void write_new_header_codes(const char *filename, const char *vin, const char *immo)
{
    FILE *bin;
    int addr, found_addr;
    unsigned char data[2], id;
    short length, i, found_length;

    bin = fopen( filename, "r+b" );
    
    if( bin != NULL )
    {
        /* Change VIN */
        addr = binary_length-1;
        id = 0x00;
        found_addr = 0;
        while( addr > (binary_length-0x300) )
        {
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            length = data[0];
            if( length == 0xFF || length == 0x00 ) break;
            addr--;
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            id = data[0];
            addr--;
            if( id == 0x90 )
            {
                found_addr = addr;
                found_length = length;
            }
            addr -= length;
        }
        if( found_addr != 0 )
        {
            for( i = 0; i < found_length; i++ )
            {
                 fseek( bin, found_addr, SEEK_SET );
                 found_addr--;
                 if( i < strlen(vin) )
                 {
                     fwrite( vin+i, sizeof(unsigned char), 1, bin );
                     //printf("addr 0x%05X = 0x%02X (%c)\r\n", addr+1, *(vin+i), *(vin+i));
                 }
                 else
                 {
                     fwrite( " ", sizeof(unsigned char), 1, bin );
                     //printf("addr 0x%05X = \" \"\r\n", addr+1);
                 }
            }
        }
        
        /* Change Immobilizer code */
        addr = binary_length-1;
        id = 0x00;
        found_addr = 0;
        while( addr > (binary_length-0x300) )
        {
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            length = data[0];
            if( length == 0xFF || length == 0x00 ) break;
            addr--;
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            id = data[0];
            addr--;
            if( id == 0x92 )
            {
                found_addr = addr;
                found_length = length;
            }
            addr -= length;
        }
        if( found_addr != 0 )
        {
            for( i = 0; i < found_length; i++ )
            {
                 fseek( bin, found_addr, SEEK_SET );
                 found_addr--;
                 if( i < strlen(immo) )
                 {
                     fwrite( immo+i, sizeof(unsigned char), 1, bin );
                     //printf("addr 0x%05X = 0x%02X (%c)\r\n", addr+1, *(immo+i), *(immo+i));
                 }
                 else
                 {
                     fwrite( " ", sizeof(unsigned char), 1, bin );
                     //printf("addr 0x%05X = \" \"\r\n", addr+1);
                 }
            }
        }


        
        fclose( bin );
    }
    else
    {
        printf("Error opening: %s\r\n", filename);
    }
            

    return;
}

void write_new_checksums(const char *filename, unsigned int fb,
                                               unsigned int f2,
                                               unsigned int misc,
                                               unsigned int area_70000)
{
    FILE *bin;
    int addr;
    unsigned char data[2], id;
    short length, i;

    bin = fopen( filename, "r+b" );
    
    if( bin != NULL )
    {
        /* Change FB */
        addr = binary_length-1;
        id = 0x00;
        while( 1 )
        {
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            length = data[0];
            if( length == 0xFF || length == 0x00 ) break;
            addr--;
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            id = data[0];
            addr--;
            if( id == 0xFB ) break;
            addr -= length;
        }
        if( id == 0xFB )
        {
            for( i = 0; i < 4; i++ )
            {
                 fseek( bin, addr, SEEK_SET );
                 addr--;
                 data[0] = (unsigned char)(fb >> (24-i*8)); 
                 fwrite( &data[0], sizeof(unsigned char), 1, bin );
                 //printf("addr 0x%05X = 0x%02X (%c)\r\n", addr+1, *(vin+i), *(vin+i));
            }
        }
        
        /* Change F2 */
        addr = binary_length-1;
        id = 0x00;
        while( 1 )
        {
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            length = data[0];
            if( length == 0xFF || length == 0x00 ) break;
            addr--;
            fseek( bin, addr, SEEK_SET );
            fread( data, sizeof(data[0]), 1, bin );
            id = data[0];
            addr--;
            if( id == 0xF2 ) break;
            addr -= length;
        }
        if( id == 0xF2 )
        {
            for( i = 0; i < 4; i++ )
            {
                 fseek( bin, addr, SEEK_SET );
                 addr--;
                 data[0] = (unsigned char)(f2 >> (24-i*8)); 
                 fwrite( &data[0], sizeof(unsigned char), 1, bin );
                 //printf("addr 0x%05X = 0x%02X (%c)\r\n", addr+1, *(immo+i), *(immo+i));
            }
        }

        /* Change Misc */
        addr = checksum_addr.misc;
        for( i = 0; i < 4; i++ )
        {
             fseek( bin, addr, SEEK_SET );
             addr++;
             data[0] = (unsigned char)(misc >> (24-i*8)); 
             fwrite( &data[0], sizeof(unsigned char), 1, bin );
        }

        /* Change Area 70000 */
        if( area_70000 != 0xFFFFFFFF )
        {
            addr = checksum_addr.area_70000;
            for( i = 0; i < 4; i++ )
            {
                 fseek( bin, addr, SEEK_SET );
                 addr++;
                 data[0] = (unsigned char)(area_70000 >> (24-i*8)); 
                 fwrite( &data[0], sizeof(unsigned char), 1, bin );
            }
        }
        
        fclose( bin );
    }
    else
    {
        printf("Error opening: %s\r\n", filename);
    }
            

    return;
}


int Symbol2Numeric(unsigned char symbol);
int MultiplyNumeric(short i, unsigned char symbol);

int check_vin( const char* vin )
{
	short i;
	int SymbolTemp;
	int SumSymbol;

	SumSymbol = 0;
	for (i=0; i < 17; i++) 
	{
		if (i==8) i++;

        if( vin[i] >= '0' && vin[i] <= '9' )
        {
            SymbolTemp = vin[i] - '0';
        }
        else
        {
        	SymbolTemp = Symbol2Numeric( vin[i] );
        }
		if( SymbolTemp == -1 ) return -1;
		SumSymbol += MultiplyNumeric( i, SymbolTemp );
    }
		
	SumSymbol = SumSymbol % 11;
	
	if( vin[8] == 'X' && SumSymbol == 10 )
	{
        return 0;
    }
    else if( (vin[8] - '0') == SumSymbol )
    {
        return 0;
    }
    else
    {
        //printf("vin[8] = %c, vin[8] - '0' = %d, SumSymbol = %d\r\n", vin[8], vin[8]-'0', SumSymbol);
        return -1;
    }
}

int Symbol2Numeric(unsigned char symbol)
{
	switch (symbol)
	{
		case 'A':
		case 'J':
			return(1);
		case 'B':
		case 'K':
		case 'S':
			return(2);
		case 'C':
		case 'L':
		case 'T':
			return(3);
		case 'D':
		case 'M':
		case 'U':
			return(4);
		case 'E':
		case 'N':
		case 'V':
			return(5);
		case 'F':
		case 'W':
			return(6);
		case 'G':
		case 'P':
		case 'X':
			return(7);
		case 'H':
		case 'Y':
			return(8);
		case 'R':
		case 'Z':
			return(9);		
		default:
			return(-1);
	}
}

int MultiplyNumeric(short i, unsigned char symbol)
{
	switch (i)
	{
		case 1:
			return(symbol*8);
		case 2:
			return(symbol*7);
		case 3:
			return(symbol*6);
		case 4:
			return(symbol*5);
		case 5:
			return(symbol*4);
		case 6:
			return(symbol*3);
		case 7:
			return(symbol*2);
		case 8:
			return(symbol*10);
		case 11:
			return(symbol*8);
		case 12:
			return(symbol*7);
		case 13:
			return(symbol*6);
		case 14:
			return(symbol*5);
		case 15:
			return(symbol*4);
		case 16:
			return(symbol*3);
		case 17:
			return(symbol*2);
		case 10:
			return(symbol*9);
		default:
			return(-1);
	}
}
